<?php

namespace App\Http\Controllers\SuperAdmin;

use App\Exports\SuperAdmin\TherapistPodcastsExport;
use App\Http\Controllers\Controller;
use App\Http\Requests\SuperAdmin\TherapistPodcasts\CreateRequest;
use App\Http\Requests\ImportRequest;
use App\Imports\SuperAdmin\TherapistPodcastsImport;
use App\Models\Podcast;
use App\Models\Therapist;
use App\Models\Tag;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Maatwebsite\Excel\Facades\Excel;
use Illuminate\Support\Str;

class TherapistPodcastsController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    /********* Initialize Permission based Middlewares  ***********/
    public function __construct()
    {
        $this->middleware('permission:therapist.add_podcast');
        $this->middleware('permission:therapist.add_podcast', ['only' => ['store']]);
        $this->middleware('permission:therapist.add_podcast', ['only' => ['update']]);
        $this->middleware('permission:therapist.add_podcast', ['only' => ['destroy']]);
        $this->middleware('permission:therapist.add_podcast', ['only' => ['export']]);
        $this->middleware('permission:therapist.add_podcast', ['only' => ['import']]);
    }
    /********* Getter For Pagination, Searching And Sorting  ***********/
    public function getter($req = null, $export = null, $therapist)
    {
        if ($req != null) {
            $therapist_podcasts =  $therapist->therapist_podcasts();
            if ($req->trash && $req->trash == 'with') {
                $therapist_podcasts =  $therapist_podcasts->withTrashed();
            }
            if ($req->trash && $req->trash == 'only') {
                $therapist_podcasts =  $therapist_podcasts->onlyTrashed();
            }
            if ($req->column && $req->column != null && $req->search != null) {
                $therapist_podcasts = $therapist_podcasts->whereLike($req->column, $req->search);
            } else if ($req->search && $req->search != null) {

                $therapist_podcasts = $therapist_podcasts->whereLike(['name', 'description'], $req->search);
            }
            if ($req->sort_field != null && $req->sort_type != null) {
                $therapist_podcasts = $therapist_podcasts->OrderBy($req->sort_field, $req->sort_type);
            } else {
                $therapist_podcasts = $therapist_podcasts->OrderBy('id', 'desc');
            }
            if ($export != null) { // for export do not paginate
                $therapist_podcasts = $therapist_podcasts->get();
                return $therapist_podcasts;
            }
            $therapist_podcasts = $therapist_podcasts->get();
            return $therapist_podcasts;
        }
        $therapist_podcasts = $therapist->therapist_podcasts()->withAll()->orderBy('id', 'desc')->get();
        return $therapist_podcasts;
    }


    /*********View All TherapistPodcasts  ***********/
    public function index(Request $request, Therapist $therapist)
    {
        $therapist_podcasts = $this->getter($request, null, $therapist);
        return view('super_admins.therapists.therapist_podcasts.index', compact('therapist_podcasts', 'therapist'));
    }

    /*********View Create Form of Podcast  ***********/
    public function create(Therapist $therapist)
    {
        $tags = Tag::active()->get();
        return view('super_admins.therapists.therapist_podcasts.create', compact('therapist', 'tags'));
    }

    /*********Store Podcast  ***********/
    public function store(CreateRequest $request, Therapist $therapist)
    {
        $data = $request->all();
        try {
            DB::beginTransaction();
            if (!$request->is_active) {
                $data['is_active'] = 0;
            }
            if ($request->link_type == 'internal' && $request->file_type == 'audio') {
                $data['audio'] = uploadFile($request, 'file', 'therapist_podcasts');
            } else {
                $data['video'] = uploadFile($request, 'file', 'therapist_podcasts');
            }
            //$data['image'] = uploadCroppedFile($request,'image','therapist_podcasts');
            $therapist_podcast = $therapist->therapist_podcasts()->create($data);
            $therapist_podcast->slug = Str::slug($therapist_podcast->name . ' ' . $therapist_podcast->id, '-');
            $therapist_podcast->save();
            $therapist_podcast->tags()->sync($request->tag_ids);
            DB::commit();
        } catch (\Exception $e) {
            DB::rollback();
            return redirect()->route('super_admin.therapist_podcasts.index', $therapist->id)->with('message', 'Something Went Wrong')->with('message_type', 'error');
        }
        return redirect()->route('super_admin.therapist_podcasts.index', $therapist->id)->with('message', 'Podcast Created Successfully')->with('message_type', 'success');
    }

    /*********View Podcast  ***********/
    public function show(Therapist $therapist, Podcast $therapist_podcast)
    {
        if ($therapist->id != $therapist_podcast->therapist_id) {
            return redirect()->back()->with('message', 'Podcast Not Found')->with('message_type', 'error');
        }
        return view('super_admins.therapists.therapist_podcasts.show', compact('therapist_podcast', 'therapist'));
    }

    /*********View Edit Form of Podcast  ***********/
    public function edit(Therapist $therapist, Podcast $therapist_podcast)
    {
        if ($therapist->id != $therapist_podcast->therapist_id) {
            return redirect()->back()->with('message', 'Podcast Not Found')->with('message_type', 'error');
        }
        return view('super_admins.therapists.therapist_podcasts.edit', compact('therapist_podcast', 'therapist'));
    }

    /*********Update Podcast  ***********/
    public function update(CreateRequest $request, Therapist $therapist, Podcast $therapist_podcast)
    {
        if ($therapist->id != $therapist_podcast->therapist_id) {
            return redirect()->back()->with('message', 'Podcast Not Found')->with('message_type', 'error');
        }
        $data = $request->all();
        try {
            DB::beginTransaction();
            if (!$request->is_active) {
                $data['is_active'] = 0;
            }
            if ($request->link_type == 'internal' && $request->file_type == 'audio') {
                if ($request->file) {
                    $data['audio'] = uploadFile($request, 'file', 'therapist_podcasts');
                } else {
                    $data['audio'] = $therapist_podcast->audio;
                }
            } else {
                if ($request->file) {
                    $data['video'] = uploadFile($request, 'file', 'therapist_podcasts');
                } else {
                    $data['video'] = $therapist_podcast->video;
                }
            }
            $therapist_podcast->update($data);
            $therapist_podcast = Podcast::find($therapist_podcast->id);
            $slug = Str::slug($therapist_podcast->name . ' ' . $therapist_podcast->id, '-');
            $therapist_podcast->update([
                'slug' => $slug
            ]);
            $therapist_podcast->tags()->sync($request->tag_ids);
            DB::commit();
        } catch (\Exception $e) {
            DB::rollback();
            return redirect()->route('super_admin.therapist_podcasts.index', $therapist->id)->with('message', 'Something Went Wrong')->with('message_type', 'error');
        }
        return redirect()->route('super_admin.therapist_podcasts.index', $therapist->id)->with('message', 'Podcast Updated Successfully')->with('message_type', 'success');
    }

    /********* Export  CSV And Excel  **********/
    public function export(Request $request)
    {
        $therapist_podcasts = $this->getter($request, "export");
        if (in_array($request->export, ['csv,xlsx'])) {
            $extension = $request->export;
        } else {
            $extension = 'xlsx';
        }
        $filename = "therapist_podcasts." . $extension;
        return Excel::download(new TherapistPodcastsExport($therapist_podcasts), $filename);
    }
    /********* Import CSV And Excel  **********/
    public function import(ImportRequest $request)
    {
        $file = $request->file('file');
        Excel::import(new TherapistPodcastsImport, $file);
        return redirect()->back()->with('message', 'Podcast Categories imported Successfully')->with('message_type', 'success');
    }


    /*********Soft DELETE Podcast ***********/
    public function destroy(Therapist $therapist, Podcast $therapist_podcast)
    {
        if ($therapist->id != $therapist_podcast->therapist_id) {
            return redirect()->back()->with('message', 'Podcast Not Found')->with('message_type', 'error');
        }
        $therapist_podcast->delete();
        return redirect()->back()->with('message', 'Podcast Deleted Successfully')->with('message_type', 'success');
    }

    /*********Permanently DELETE Podcast ***********/
    public function destroyPermanently(Request $request, Therapist $therapist, $therapist_podcast)
    {
        $therapist_podcast = Podcast::withTrashed()->find($therapist_podcast);
        if ($therapist_podcast) {
            if ($therapist_podcast->trashed()) {
                if ($therapist_podcast->image && file_exists(public_path($therapist_podcast->image))) {
                    unlink(public_path($therapist_podcast->image));
                }
                $therapist_podcast->forceDelete();
                return redirect()->back()->with('message', 'Podcast Deleted Successfully')->with('message_type', 'success');
            } else {
                return redirect()->back()->with('message', 'Podcast is Not in Trash')->with('message_type', 'error');
            }
        } else {
            return redirect()->back()->with('message', 'Podcast Not Found')->with('message_type', 'error');
        }
    }
    /********* Restore Podcast***********/
    public function restore(Request $request, Therapist $therapist, $therapist_podcast)
    {
        $therapist_podcast = Podcast::withTrashed()->find($therapist_podcast);
        if ($therapist_podcast->trashed()) {
            $therapist_podcast->restore();
            return redirect()->back()->with('message', 'Podcast Restored Successfully')->with('message_type', 'success');
        } else {
            return redirect()->back()->with('message', 'Podcast Not Found')->with('message_type', 'error');
        }
    }
}
